* Title: 	table11.do
* Version: 	23 May 2018
* Purpose: 	Produce table of model calibration for Wyoming/West Virginia/Massachusetts

*******************************************************************************
* (0) Start of file
*******************************************************************************

capture log close
log using output/tables/log/table11, text replace
set more off
clear all
macro drop _all


*******************************************************************************
* (1) Generate aggregate version of Table 6
*******************************************************************************

use year statefip RegionI PrimeMaleD FamilyI ///
	EmploymentI Inc_* wtsupp ///
	using dta/asec_clean, clear

* Keep only prime-age men
keep if PrimeMaleD == 1

* Keep data for 2014 to 2016
keep if year>=2010 & year<=2016

* Exclude cases with negative total income
drop if (Inc_FamilyTotal < 0 | Inc_PersonalTotal < 0)

* Group 1: short-term jobless, all
gen group1 = (EmploymentI == 1)

* Group 2: long-term jobless, all
gen group2 = (EmploymentI == 2)

* Group 3: all jobless
gen group3 = (EmploymentI == 1 | EmploymentI == 2)


forval i = 1/3 {
	foreach var of varlist Inc_FamilyTotal-Inc_Other {
		assert !mi(`var')
		gen `var'`i' = `var' if (group`i' == 1)
	}
	
	gen Total`i' = PrimeMaleD if (group`i' == 1)
}

collapse 	(sum)	Total = PrimeMaleD Total* ///
			(mean)	Inc_*1 Inc_*2 Inc_*3 ///
			[pw = wtsupp]
			
forval i = 1/3 {
	gen Group`i'l_GovernmentOther = Inc_GovernmentTotal`i' - Inc_Unemp`i' ///
									- Inc_Disability`i' - Inc_Vet`i'			  
	gen Group`i'm_Other_Inc_Round = Inc_PersonalTotal`i' - Inc_Wage`i' ///
									- Inc_Investment`i' - Inc_Retirement`i' ///
									- Inc_WorkersComp`i' - Inc_FamilyTransfer`i' ///
									- Inc_GovernmentTotal`i'
	gen Group`i'n_Share = Total`i'/Total

					
	ren Inc_FamilyTotal`i' 		Group`i'a_FamilyTotal
	ren Inc_PersonalTotal`i'	Group`i'b_PersonalTotal
	ren Inc_Wage`i'				Group`i'c_Wage
	ren Inc_Investment`i'		Group`i'd_Investment
	ren Inc_Retirement`i'		Group`i'e_Retirement
	ren Inc_WorkersComp`i'		Group`i'f_WorkersComp
	ren Inc_FamilyTransfer`i'	Group`i'g_FamilyTransfer
	ren Inc_GovernmentTotal`i'	Group`i'h_GovernmentTotal
	ren Inc_Unemp`i'			Group`i'i_Unemp
	ren Inc_Disability`i'		Group`i'j_Disability
	ren Inc_Vet`i'				Group`i'k_Vet
	
	drop Total`i' Inc_GovernmentOther`i' Inc_Other`i'
}
reshape long Group1 Group2 Group3, i(Total) j(category) str
drop Total

replace category = "Total family income" 		if category == "a_FamilyTotal"
replace category = "Total individual income" 	if category == "b_PersonalTotal"
replace category = "Wage" 						if category == "c_Wage"
replace category = "Investment/Business" 		if category == "d_Investment"
replace category = "Retirement" 				if category == "e_Retirement"
replace category = "Workers' compensation" 		if category == "f_WorkersComp"
replace category = "Family transfers" 			if category == "g_FamilyTransfer"
replace category = "Total government support" 	if category == "h_GovernmentTotal"
replace category = "Unemployment" 				if category == "i_Unemp"
replace category = "Disability" 				if category == "j_Disability"
replace category = "Veterans benefits" 			if category == "k_Vet"
replace category = "Other government support"	if category == "l_GovernmentOther"
replace category = "Other sources" 				if category == "m_Other_Inc_Round"
replace category = "Share of prime-age men" 	if category == "n_Share"

ren Group1 STJobless
ren Group2 LTJobless
ren Group3 AllJobless

tempfile table6_agg
save `table6_agg'


*******************************************************************************
* (2) Calculate externality ratio
*******************************************************************************

* Disability and government support
use `table6_agg', clear
summ AllJobless if category == "Total government support"
local govt_alljobless `r(mean)'
summ AllJobless if category == "Disability"
local disab_alljobless `r(mean)'

use output/tables/dta/table6.dta, clear
summ Employed_LowIncome if category == "Total government support"
local govt_empl_low `r(mean)'

local disab_gov_support_ext = round((`govt_alljobless' - `govt_empl_low') - 0.5*`disab_alljobless',100)

* Tax externality
use output/tables/dta/table8.dta, clear
gen diff = LTJobless_LiveAlone - Employed_LiveAlone_LowIncome
summ diff if category == "Tax"
local tax_ext = -round(`r(mean)',100)

* Family externality
use output/tables/dta/table8.dta, clear
summ LTJobless_LiveAlone if category == "Total expenditure"
local totexp `r(mean)'
summ LTJobless_LiveAlone if category == "Housing"
local housexp `r(mean)'
local family_ext = round(0.25 * (`totexp' - `housexp'),100)

* Total externality
local total_ext = `disab_gov_support_ext' + `tax_ext' + `family_ext'

* Low-income wage
use output/tables/dta/table6.dta, clear
summ Employed_LowIncome if category == "Wage"
local wage_empl_low `r(mean)'

* Ratio of total externality to low-income wage
local externality_ratio round(`total_ext'/`wage_empl_low',0.001)


*******************************************************************************
* (3) Compute state-level elasticities from results of Table 10
*******************************************************************************

use output/tables/dta/table10.dta, clear

* Column 2: IV, excluding state fixed effects
ivreg2 LogEmpRate (log_incwage_p25 IncWage_p25_IntJob = Bartik Bartik_Sq Bartik_Cu Bartik_IntJob Bartik_IntJob_Sq Bartik_IntJob_Cu) ///
					Jobless_Rate_1980 i.year College_Rate_1980 LessHSEd_Rate_1980, ///
					first cluster(statefip)

local coeff_wage = _b[log_incwage_p25]
di `coeff_wage'
local coeff_interact = _b[IncWage_p25_IntJob]
di `coeff_interact'

* Get 1980 jobless rate by state from Census
use year statefip UnempB PrimeMaleD perwt using dta/census_clean, clear
keep if year == 1980
keep if PrimeMaleD == 1
collapse (sum) Total = PrimeMaleD Jobless = UnempB [pw = perwt], by(statefip)
gen Jobless_Rate1980 = Jobless/Total

* Calculate state-level elasticities
gen elasticity = `coeff_wage' + `coeff_interact'* Jobless_Rate1980

keep statefip elasticity
tempfile elast
save `elast'


*******************************************************************************
* (4) Create Table 11
*******************************************************************************

* Get at-risk jobless rate from 2014-16 ASEC
use year statefip PrimeMaleD UnempB EmploymentI Inc_FamilyTotal Inc_PersonalTotal wtsupp /// 
		using dta/asec_clean, clear
keep if year>=2014 & year<=2016
keep if PrimeMaleD == 1
drop if (Inc_FamilyTotal < 0 | Inc_PersonalTotal < 0)
keep if  (EmploymentI == 0 & Inc_FamilyTotal < 40000) | UnempB == 1
collapse (sum) Total = PrimeMaleD Jobless = UnempB [pw = wtsupp], by(statefip)
gen double At_Risk_Rate = Jobless/Total

* Elasticity
keep statefip At_Risk_Rate
merge 1:1 statefip using `elast', assert(matched) nogen

* Externality ratio
gen externality_ratio = `externality_ratio'

* Calculate consumption ratio for different values of gamma
gen Gamma1 = 0.5
gen Gamma2 = 1
gen Gamma3 = 2

gen ConsRatio = .
forval i =1/3 {
	gen ConsRatio`i' = (1 - ((elasticity/At_Risk_Rate) * externality_ratio))^(1/Gamma`i')
}

* Keep only Wyoming, West Virginia, and Massachusetts
keep if statefip == 56 | statefip == 54 | statefip == 25

* Construct table
keep statefip At_Risk_Rate elasticity externality_ratio ConsRatio*
foreach var of varlist At_Risk_Rate externality_ratio ConsRatio1 ConsRatio2 ConsRatio3 {
	replace `var' = 100 * `var'
}
ren At_Risk_Rate		Va_At_Risk_Rate
ren elasticity			Vb_elasticity
ren externality_ratio	Vc_externality_ratio
ren ConsRatio			Vd_ConsRatio
ren ConsRatio1			Ve_ConsRatio1
ren ConsRatio2			Vf_ConsRatio2
ren ConsRatio3			Vg_ConsRatio3

gen i = 1
reshape wide V*, i(i) j(statefip)
ren V*25 MA*
ren V*54 WV*
ren V*56 WY*
reshape long MA WV WY, i(i) j(category) str
drop i

replace category = "At-risk Not Working Rate (2014-2016)"	if category == "a_At_Risk_Rate"
replace category = "Elasticity of the employment rate"		if category == "b_elasticity"
replace category = "Externality as percent of wage"			if category == "c_externality_ratio"
replace category = "Ratio of consumption"					if category == "d_ConsRatio"
replace category = "γ = 0.5"								if category == "e_ConsRatio1"
replace category = "γ = 1"									if category == "f_ConsRatio2"
replace category = "γ = 2"									if category == "g_ConsRatio3"

order category WY WV MA
ren WY Wyoming
ren WV West_Virginia
ren MA Massachusetts

* Save dataset
save output/tables/dta/table11.dta, replace


*******************************************************************************
* (5) End of file
*******************************************************************************

log close
exit, clear

